const WIDGET_WIDTH_CHAT_FEED = 445;
const WIDGET_WIDTH_CHAT_LIST = 353;
const WIDGET_WIDTH_CHAT_DATA_PANEL = 295;

const RESIZABLE_TRANSITION_DURATION = 400;
const RESIZABLE_TRANSITION = `transform ${RESIZABLE_TRANSITION_DURATION}ms cubic-bezier(.4, 0, .2, 1)`;

let overlay;
let resizable;
let iframe;

let currentPointerdownEvent;

let iframeWindow = null;
let handshakeId = null;
let isResizing = false;
let adjustSizeTimeout = null;

function setHandshakeParams(event) {
  if (
    !isObject(event.data)
    || event.data.action !== ACTION_HANDSHAKE
    || !isNotEmptyString(event.data.handshakeId)
  ) {
    return;
  }

  if (!isObject(iframeWindow)) {
    iframeWindow = event.source;
  }

  handshakeId = event.data.handshakeId;
}

function isIframeMessage(event) {
  return event.source === iframeWindow
    && event.origin === ROISTAT_HOST
    && isObject(event.data)
    && isNotEmptyString(event.data.action)
    && isObject(event.data.detail)
    && isNotEmptyString(event.data.handshakeId)
    && handshakeId === event.data.handshakeId;
}

function isExtensionMessage(message) {
  return isObject(message)
    && isNotEmptyString(message.action)
    && isObject(message.detail);
}

function postMessage(action, detail = {}) {
  if (
    isObject(iframeWindow)
    && isNotEmptyString(action)
    && isObject(detail)
  ) {
    iframeWindow.postMessage({ action, detail, handshakeId }, ROISTAT_HOST);
  }
}

async function cleanUp() {
  if (!isHtmlElement(overlay)) {
    return;
  }

  handleMouseUp();

  postMessage(ACTION_CLOSE_CHAT_EXTENSION);

  await new Promise(resolve => setTimeout(() => {
    overlay.remove();
    resolve();
  }, 10));
}

function handleClose() {
  if (!isHtmlElement(overlay) || !isHtmlElement(resizable)) {
    return;
  }

  isResizing = false;

  overlay.style.opacity = '0';
  resizable.style.transform = 'translateX(100%)';

  setTimeout(async () => {
    await cleanUp();

    iframeWindow = null;
  }, RESIZABLE_TRANSITION_DURATION);
}

function handleAdjustSize({ hasChatList, hasChatDataPanel }) {
  if (!isHtmlElement(resizable)) {
    return;
  }

  resizable.style.transition = `width ${RESIZABLE_TRANSITION_DURATION}ms ease-in-out`;
  clearTimeout(adjustSizeTimeout);
  adjustSizeTimeout = setTimeout(() => {
    if (isHtmlElement(resizable)) {
      resizable.style.transition = RESIZABLE_TRANSITION;
    }
  }, RESIZABLE_TRANSITION_DURATION);

  const newWidth = WIDGET_WIDTH_CHAT_FEED
    + (hasChatList ? WIDGET_WIDTH_CHAT_LIST : 0)
    + (hasChatDataPanel ? WIDGET_WIDTH_CHAT_DATA_PANEL : 0);
  resizable.style.width = `${newWidth}px`;
}

function handleMouseMove(event) {
  if (!isResizing || !isHtmlElement(resizable)) {
    return;
  }

  const newWidth = Math.min(
    Math.max(window.innerWidth - event.clientX, WIDGET_WIDTH_CHAT_FEED),
    window.innerWidth * 0.9
  );
  resizable.style.width = `${newWidth}px`;
}

function handleMouseUp() {
  if (!isHtmlElement(resizable)) {
    return;
  }

  isResizing = false;

  document.removeEventListener('pointermove', handleMouseMove);
  document.removeEventListener('pointerup', handleMouseUp);
  try {
    resizable.releasePointerCapture(currentPointerdownEvent.pointerId);
  } catch(e) {}
}

function handleResize(pointerdownEvent) {
  currentPointerdownEvent = pointerdownEvent;
  try {
    currentPointerdownEvent.target.setPointerCapture(currentPointerdownEvent.pointerId);
  } catch(e) {}
  isResizing = true;

  document.addEventListener('pointermove', handleMouseMove);
  document.addEventListener('pointerup', handleMouseUp);
}

function createOverlay() {
  overlay = document.createElement('div');
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    width: 100vw;
    height: 100vh;
    backdrop-filter: blur(4px);
    z-index: 999999;
    display: flex;
    justify-content: flex-end;
    align-items: stretch;
    opacity: 0;
    transition: opacity ${RESIZABLE_TRANSITION_DURATION}ms ease;
  `;

  document.body.appendChild(overlay);

  overlay.addEventListener('mousedown', event => {
    if (event.target === overlay) {
      event.stopPropagation();
      event.preventDefault();
      handleClose();
    }
  });

  requestAnimationFrame(() => {
    if (isHtmlElement(overlay)) {
      overlay.style.opacity = '1';
    }
  });
}

function createResizable() {
  if (!isHtmlElement(overlay)) {
    return;
  }

  resizable = document.createElement('div');
  resizable.style.cssText = `
    position: relative;
    width: ${WIDGET_WIDTH_CHAT_FEED}px;
    height: 100vh;
    background: #fff;
    box-shadow: -2px 0 8px rgba(0, 0, 0, .1);
    transform: translateX(100%);
    transition: ${RESIZABLE_TRANSITION};
  `;

  overlay.appendChild(resizable);

  requestAnimationFrame(() => {
    if (isHtmlElement(resizable)) {
      resizable.style.transform = 'translateX(0)';
    }
  });
}

function createResizer() {
  if (!isHtmlElement(resizable)) {
    return;
  }

  const resizer = document.createElement('div');
  resizer.style.cssText = `
    position: absolute;
    top: 0;
    left: -2px;
    min-width: 4px;
    max-width: 4px;
    height: 100%;
    cursor: ew-resize;
  `;

  resizer.addEventListener('pointerdown', event => handleResize(event));

  resizable.appendChild(resizer);
}

function createIframe(url) {
  if (!isHtmlElement(resizable)) {
    return;
  }

  iframe = document.createElement('iframe');
  iframe.src = url;
  iframe.style.cssText = `
    width: 100%;
    height: 100%;
    border: none;
  `;
  iframe.setAttribute('sandbox', 'allow-same-origin allow-scripts allow-popups allow-downloads allow-modals');
  iframe.setAttribute('title', 'Roistat Чат');

  resizable.appendChild(iframe);
}

window.addEventListener('message', event => {
  setHandshakeParams(event);

  if (!isIframeMessage(event)) {
    return;
  }

  switch (event.data.action) {
    case ACTION_HANDSHAKE: {
      postMessage(ACTION_HANDSHAKE_RESPONSE);
      break;
    }

    case ACTION_CLOSE_CHAT_EXTENSION: {
      handleClose();
      break;
    }

    case ACTION_ADJUST_EXTENSION_SIZE: {
      handleAdjustSize(event.data.detail);
      break;
    }
  }
});

chrome.runtime.onMessage.addListener(async message => {
  if (!isExtensionMessage(message)) {
    return;
  }

  if (message.action === ACTION_OPEN_CHAT_EXTENSION) {
    if (isHtmlElement(overlay)) {
      await cleanUp();
    }

    createOverlay();
    createResizable();
    createResizer();
    createIframe(message.detail.url);
  }
});
