importScripts('const.js');
importScripts('validate.js');

const OPEN_ROISTAT_CHAT_MENU_ITEM = 'openRoistatChat';

const ROISTAT_CHAT_URL = `${ROISTAT_HOST}/chat`;
const ROISTAT_PHONE_URL = `${ROISTAT_CHAT_URL}?isExtension=1&clientPhone=`;
const ROISTAT_EMAIL_URL = `${ROISTAT_CHAT_URL}?isExtension=1&clientEmail=`;

const PHONE_REGEX = /[^0-9+]/g;
const EMAIL_REGEX = /^[a-zA-Z0-9._%+-]*@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;

function getFormattedPhoneNumber(input) {
  const result = input.replace(PHONE_REGEX, '');

  return isNotEmptyString(result) ? result : null;
}

function getFormattedEmail(string) {
  return EMAIL_REGEX.test(string)
    ? encodeURIComponent(string.toLowerCase())
    : null;
}

function getRoistatChatUrl(email, phone) {
  if (isNotEmptyString(email)) {
    return ROISTAT_EMAIL_URL + email;
  }

  if (isNotEmptyString(phone)) {
    return ROISTAT_PHONE_URL + phone;
  }

  return null;
}

function executeScript(tabId, func) {
  chrome.scripting.executeScript({
    target: { tabId },
    func,
  });
}

chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: OPEN_ROISTAT_CHAT_MENU_ITEM,
    title: 'Открыть в Roistat',
    contexts: ['selection'],
  });
});

chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (!isObject(info) || !isObject(tab) || !isNumber(tab.id) || !isNotEmptyString(tab.url)) {
    return;
  }

  const selectedButton = info.menuItemId;
  const selectedText = info.selectionText;

  if (![OPEN_ROISTAT_CHAT_MENU_ITEM].includes(selectedButton) || !isNotEmptyString(selectedText)) {
    return;
  }

  if (tab.url.startsWith(ROISTAT_CHAT_URL)) {
    executeScript(tab.id, () => {
      alert('Вы и так находитесь в онлайн-чате.');
    });
    return;
  }

  const selectedTrimmedText = selectedText.trim();

  const phone = getFormattedPhoneNumber(selectedTrimmedText);
  const email = getFormattedEmail(selectedTrimmedText);

  if (!isNotEmptyString(phone) && !isNotEmptyString(email)) {
    executeScript(tab.id, () => {
      alert('Выделенный текст не похож на номер телефона или email.');
    });
    return;
  }

  const url = getRoistatChatUrl(email, phone);

  if (isNotEmptyString(url)) {
    chrome.tabs.sendMessage(tab.id, {
      action: ACTION_OPEN_CHAT_EXTENSION,
      detail: { url },
    });
  }
});
